/*************************************************/
/* Examples Program For "ET-dsPIC33WEB-V1" Board */
/* Hardware    : ET-dsPIC33WEB-V1.0              */
/* Target MCU  : dsPIC33FJ128GP708               */
/*       	   : X-TAL : 8.00 MHz                */
/* Config Fosc : XTAL = 8MHz (XT+PLL)            */
/*             : PLLPRE[4..0] = 00000 (N1=2)     */
/*             : VCO Input = 8MHz / 2 = 4MHz     */
/*             : PLLDIV[8..0] = 0x26(M=40)       */
/*             : VCO Output = 4 x 40MHz = 160MHz */
/*             : PLLPOST[1:0] = 0:0 (N2=2)       */
/*             : Fosc = 160MHz / 2 = 80MHz       */
/*             : Fcy = Fosc/2 = 80MHz /2 = 40MHz */
/* Compiler    : MPLAB + C30 V3.01  		     */
/* Write By    : Eakachai Makarn(ETT CO.,LTD.)   */
/* Last Update : 16/August/2007                  */
/* Function    : Example Use UART = 9600 BPS     */
/*************************************************/
/* Used RF2 = RXD1                               */
/* Used RF3 = TXD1                               */
/* Used RF4 = RXD2                               */
/* Used RF5 = TXD2                               */
/*************************************************/

#include "p33FJ128GP708.h"                             				// dsPIC33FJ128GP708 MPU Register
#include "stdio.h"													// Used "sprintf" Function

/* Setup Configuration For ET-dsPIC33WEB-V1 */
_FOSCSEL(FNOSC_PRIPLL & 											// Select Primary OSC = XT + PLL
         IESO_OFF )													// Startup With User Select OSC
_FOSC(FCKSM_CSDCMD &        										// Clock Switch Disabled	
      OSCIOFNC_OFF &												// OSC2 Pin = Clock O/P
      POSCMD_XT)													// Primary OSC = XT 
_FWDT(FWDTEN_OFF &           										// Disable Watchdog Timer 
      WINDIS_OFF &          										// Disabled Windowed WDT
      WDTPRE_PR128 &        										// Watchdog prescaler=1:128
      WDTPOST_PS32768)     											// Watchdog postscaler=1:32,768
_FPOR(FPWRT_PWR128)       											// Power-on Reset Value = 128ms
_CONFIG3(JTAGEN_OFF);												// Disable JTAG Interface
/* End of Configuration For ET-dsPIC33WEB-V1 */

char uart1_buf[40]; 												// "sprint" UART1 Buffer
char uart2_buf[40]; 												// "sprint" UART2 Buffer

/* pototype  section */
void putchar1(char ch);  											// Put Char To UART-1
void putchar2(char ch);  											// Put Char To UART-2
char getchar1(void);  												// Get Char From UART-1
char getchar2(void);  												// Get Char From UART-2
void print_uart1(void);												// Print String to UART1 
void print_uart2(void);												// Print String to UART2 
void delay(unsigned long int);										// Delay Time Function

int main(void)
{  
  unsigned int uart_data;											// Character Receive Buffer

  /* Start of PLL Config Fcy = 40MIPS */
  CLKDIV &= 0xFFE0;													// PLLPRE[4..0] = 00000 (N1=2)
  CLKDIV &= 0xFF3F;													// PLLPOST[1:0] = 00 (N2=2)
  PLLFBD  = 0x26;													// PLLDIV[8..0] = 000100110(M=40)
  while(!OSCCONbits.LOCK);											// Wait PLL to Lock
  OSCCONbits.CLKLOCK = 1;											// Disable PLL Modify
  /* End of PLL Config Fly = 40MIPS */

  /* Confog UART[1] */	
  U1MODEbits.UARTEN   = 1;											// Enable RXD UART
  U1MODEbits.USIDL    = 0;											// UART Continue in IDLE
  U1MODEbits.IREN     = 0;											// IRDA Disable
  U1MODEbits.RTSMD    = 0;											// RTS = Flow Control		
  U1MODEbits.UEN1     = 0;											// Enable RX,TX Pin
  U1MODEbits.UEN0     = 0;											
  U1MODEbits.WAKE     = 0;											// Disable Wakeup
  U1MODEbits.LPBACK   = 0;											// Disable Loop Back
  U1MODEbits.ABAUD    = 0;											// Disable Auto Baudrate
  U1MODEbits.URXINV   = 0;											// RXD Pin in IDLE = "1"
  U1MODEbits.BRGH     = 0;											// Baudrate = 16 Cycle Clock
  U1MODEbits.PDSEL1   = 0;											// 8 Bit Data,No Parity
  U1MODEbits.PDSEL0   = 0;											
  U1MODEbits.STSEL    = 0;											// 1 Stop Bit
  U1STAbits.UTXISEL1  = 0;											// Interrupt By TX 1-Char
  U1STAbits.UTXISEL0  = 0;										
  U1STAbits.UTXINV	  = 0;											// IRDA Encode in IDLE = 0
  U1STAbits.UTXBRK	  = 0;											// Disable Sync Break
  U1STAbits.UTXEN     = 1;  										// Enable TXD UART 
  U1STAbits.URXISEL1  = 0;											// Interrupt BY RX 1-Char
  U1STAbits.URXISEL0  = 0;										
  U1STAbits.ADDEN	  = 0;											// Disable Address Detect

  // UART[1] Baudrate 
  // ET-dsPIC33WEB-V1 Hardware Board
  // Fcy = 40MHz
  // UxBRG = [Fcy/(16xBaud)]-1
  //       = [ 40MHz / (16 x 9600) ] - 1
  //       = 259 
  U1BRG  = 259;														// UART1 Baudrate 9600/40MHz

  // UART[1] Interrupt Control 
  IEC0bits.U1RXIE     = 0;											// Disable RX Interrupt
  IEC0bits.U1TXIE     = 0;											// Disable TX Interrupt

  /* Config UART[2] */
  U2MODEbits.UARTEN   = 1;											// Enable RXD UART
  U2MODEbits.USIDL    = 0;											// UART Continue in IDLE
  U2MODEbits.IREN     = 0;											// IRDA Disable
  U2MODEbits.RTSMD    = 0;											// RTS = Flow Control		
  U2MODEbits.UEN1     = 0;											// Enable RX,TX Pin
  U2MODEbits.UEN0     = 0;											
  U2MODEbits.WAKE     = 0;											// Disable Wakeup
  U2MODEbits.LPBACK   = 0;											// Disable Loop Back
  U2MODEbits.ABAUD    = 0;											// Disable Auto Baudrate
  U2MODEbits.URXINV   = 0;											// RXD Pin in IDLE = "1"
  U2MODEbits.BRGH     = 0;											// Baudrate = 16 Cycle Clock
  U2MODEbits.PDSEL1   = 0;											// 8 Bit Data,No Parity
  U2MODEbits.PDSEL0   = 0;											
  U2MODEbits.STSEL    = 0;											// 1 Stop Bit
  U2STAbits.UTXISEL1  = 0;											// Interrupt By TX 1-Char
  U2STAbits.UTXISEL0  = 0;										
  U2STAbits.UTXINV	  = 0;											// IRDA Encode in IDLE = 0
  U2STAbits.UTXBRK	  = 0;											// Disable Sync Break
  U2STAbits.UTXEN     = 1;  										// Enable TXD UART 
  U2STAbits.URXISEL1  = 0;											// Interrupt BY RX 1-Char
  U2STAbits.URXISEL0  = 0;										
  U2STAbits.ADDEN	  = 0;											// Disable Address Detect

  // UARTR[2] Baudrate
  // ET-dsPIC33WEB-V1 Hardware Board
  // Fcy = 40MHz
  // UxBRG = [Fcy/(16xBaud)]-1
  //       = [ 40MHz / (16 x 9600) ] - 1
  //       = 259 
  U2BRG = 259;														// UART2 Baudrate 9600/40MHz

  // UART[2] Interrupt Control 
  IEC1bits.U2RXIE = 0;												// Disable RX Interrupt
  IEC1bits.U2TXIE = 0;												// Disable TX Interrupt

  // Wait Power-On Delay //
  delay(1000000);

  // UART[1] Print String //
  sprintf(uart1_buf,"\n\r\rET-dsPIC33WEB-V1.0 : UART1\n\r"); 		// Print Message String 
  print_uart1();
 
  // UART[2] Print String //
  sprintf(uart2_buf,"\n\r\rET-dsPIC33WEB-V1.0 : UART2\n\r");		// Print Message String 
  print_uart2();

  while(1)															// Loop Continue
  {
    if(U1STAbits.URXDA)												// Check UART1 Receive
    {
      uart_data = getchar1();										// Wait Receive Byte From UART1
      if(uart_data==0x0D)											// If Enter
      {
        sprintf(uart1_buf,"\n\rET-dsPIC33WEB-V1.0 : UART[1]\n\r"); 	// Print Message String 
        print_uart1();
      }
      putchar1(uart_data);											// Echo Data to UART1
    }

    if(U2STAbits.URXDA)												// Check UART2 Receive
    {
      uart_data = getchar2();										// Wait Receive Byte From UART2
      if(uart_data==0x0D)											// If Enter
      {
        sprintf(uart2_buf,"\n\rET-dsPIC33WEB-V1.0 : UART[2]\n\r");	// Print Message String 
        print_uart2();
      }
	  putchar2(uart_data);											// Echo Data to UART2
	}
  }	
}	  

/****************************/
/* Write Character To UART1 */
/****************************/
void putchar1 (char c)  
{
  if (c == '\n')  													// If Line Feed(LF)
  {
    U1TXREG = 0x0D;                        							// Auto Add CR(LF+CR)
    while (!U1STAbits.TRMT);										// Wait Send Complete
    U1STAbits.TRMT = 0;												// Reset Status
  }
  U1TXREG = c;														// Send This Char
  while (!U1STAbits.TRMT);											// Wait Send Complete
  U1STAbits.TRMT = 0;												// Reset Status
}

/****************************/
/* Write Character To UART2 */
/****************************/
void putchar2 (char c)  
{
  if (c == '\n')  													// If Line Feed(LF)
  {
    U2TXREG = 0x0D;                        							// Auto Add CR(LF+CR)
    while (!U2STAbits.TRMT);										// Wait Send Complete
    U2STAbits.TRMT = 0;												// Reset Status
  }
  U2TXREG = c;														// Send This Char
  while (!U2STAbits.TRMT);											// Wait Send Complete
  U2STAbits.TRMT = 0;												// Reset Status
}

/****************************/
/* Get character From UART1 */
/****************************/
char getchar1 ()  
{
  char c;
  while (!U1STAbits.URXDA);											// Wait UART1 Ready
  c = U1RXREG;														// Read UART1 Data
  U1STAbits.URXDA = 0;												// Reset Status
  return (c);
}

/****************************/
/* Get character From UART2 */
/****************************/
char getchar2 ()  
{
  char c;
  while (!U2STAbits.URXDA);											// Wait UART2 Ready
  c = U2RXREG;														// Read UART2 Data
  U2STAbits.URXDA = 0;												// Reset Status
  return (c);
}

/*************************/
/* Print String to UART1 */
/*************************/
void print_uart1(void)
{
  char *p;															// Pointer Buffer
  p = uart1_buf;													// UART1 Buffer
 
  do 																// Get char & Print Until null
  {
    putchar1(*p); 													// Write char to UART1
    p++;															// Next char
  }
  while(*p != '\0');												// End of ASCII (null)
 
  return;
}

/*************************/
/* Print String to UART2 */
/*************************/
void print_uart2(void)
{
  char *p;															// Pointer Buffer
  p = uart2_buf;													// UART1 Buffer
 
  do 																// Get char & Print Until null
  {
    putchar2(*p); 													// Write char to UART2
    p++;															// Next char
  }
  while(*p != '\0');												// End of ASCII (null)
 
  return;
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}										// Loop Decrease Counter	
}
